function [Data, IR, coef] = gen_array(dim, Num_cts);
%generate data of double exponential decay with different amplitude
% dim: number of decays in data set
% Num_cts: average number of photon counts in image 


if nargin == 0
    prompt  = {'dimension of data set','total number of counts:', 'noise'};
    title_W1   = 'Array generation paramters';
    lines= 1;
    
    if exist('gen_answer', 'var') == 1
        gen_def = gen_answer;
    else
        gen_def = {'16','1e4','Yes'};
    end
    gen_answer  = inputdlg(prompt,title_W1,lines,gen_def);
    
    dim = str2num(char(gen_answer(1)));         %dimension of data set (Nb of pixels)
    Num_cts = str2num(char(gen_answer(2)));     %Intensity of the signal
    if strcmpi(gen_answer(3),'yes')
        Noise = 1;
    else
        Noise = 0;
    end
else
    Noise = 1;
end

%T1 and T2: two time constants for exp decays
T1 = 1/4e-9; 
T2 = 1/2e-9; 

% T_min_T_max: time span of data measurment
T_min = 1.2e-9;
T_max = 8.5e-9;

time = [T_min:(T_max-T_min)/46:T_max];


coef = 0;
coef(1) = T1;                       % set time constants
coef(2) = T2;

coef(3:2:dim*2+1) = round(10.*rand(dim,1))./10; %set intensity value for each pixel randomly

coef(4:2:dim*2+2) = [dim-1:-1:0]./(dim-1); %set exponential relative ratio varies linearly throughout image

G = gauss(0.25e-9, 2.3e-9, time);  %generate gaussian Impulse response

Y_data =  dblexp_conv(coef, time, G); % generates Data: an dblexponantial decay convoluted with impulse response

Y_data = Counts_Noise(Y_data, Num_cts, Noise);  %add poisson noise to the data to simulate photon counting experiment


figure(2)
plot(time, Y_data);

Data = [time; Y_data];
IR = G;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                       %
%____________________ SUB-ROUTINES _____________________%
%                                                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%_________Gaussian function_______________________

function F = gauss(FWHM, center, x)

sigma = FWHM^2/4/log(2);

for i = 1:length(x)
    F(i) = exp(-(x(i) - center)^2/sigma);
end
%sumF = sum(F)
F = F/sum(F);

%________________dblexp convolution ______________________________

function F = dblexp_conv(coef, x, G)

Nb_pix = (length(coef)-2)/2;        %Number of pixels
T_points = length(x);    %Number of time points
[Tmax, Tmax_i] = max(G);  %finds Time zero indice
L_coef = length(coef);
x_ext = x;
x_ext(T_points+1) = x(T_points) + x(2) - x(1);
x_ext = x_ext - x_ext(Tmax_i);
x_ext2D = [0:T_points+Tmax_i-1].*(x(2) - x(1)) + x(1);
x_ext2D = x_ext2D - x_ext2D(Tmax_i);
A_2D = zeros(Nb_pix, T_points + Tmax_i);

coef_3 = coef(3:2:L_coef-1);
coef_4 = coef(4:2:L_coef);

for i = Tmax_i:T_points+Tmax_i
    A_2D(:,i) = coef_3(:).*(coef_4(:).*exp(-coef(1)*x_ext2D(i)) + (1-coef_4(:)).*exp(-coef(2)*x_ext2D(i)));
    offset(:,i-Tmax_i+1) = 0.1*coef_3(:);
end

F = conv_2D(A_2D(:,Tmax_i:T_points+Tmax_i), G) + offset(:,1:T_points);


%_____________Add counts noise to the data_________________


function F = Counts_Noise(Y_Data, Num_cts, N)

L = size(Y_Data,2);           %number of time points
K = size(Y_Data,1);         %Number of pixels
Int = sum(sum(Y_Data)')/K;  %average intensity per pixel
R = Num_cts/Int;            %Multiplying ratio to reach intensity values

Decay = R.*Y_Data;
if N == 1
   Decay = poissrnd(Decay);
end

F = Decay;


%_________________conv_2D_________________

function F = conv_2D(decay_2D, IR)
% calculates the convolution between IR and each line of the decay matrix

Nb_pix = size(decay_2D,1);
T_points = length(IR);
F = zeros(Nb_pix,T_points);

for i = 1:T_points
    for k = 1:i
        F(:,i) = F(:,i) + IR(k).*decay_2D(:,i-k+1);
    end
end
