function [coef_fit, Data_fit, iter_evol] = fit_IC_global(Data, IR, partition, seg_file);
%Least square fit to find the 4 coefficients that best reproduce the Y_data matrix
%using a convolution of a given impulse  response (IR) and a double
%exponantial decay. See paper in Biophys J. for more details
%
% Data is a 2D matrix withthe first line being the time points and all the
% other lines containing the emitted photons at those times for each decay
% in the image
% IR: is the impulse response based on the same time axis
%partition determines the image analysis technique. 
%partition == 1 segmentation by intensity
%partition == 2 segmentation lifetime 
%partition == 3: fit all to get T and then fit global (image average)
%partition == 4: sum 4 parts and then global (Quadrant average)
%seg_file is a presegmented file that can be loaded direcly in the function
%use Data_segmentation to generate it and save its output(Data_seg and
%Seg_mask) in a .mat file

global ERR ITER ITER_EVOL T_START FULL_SCALE TOL
ITER = 0;
ITER_EVOL = zeros(2000,4);
T_START = cputime;
FULL_SCALE = 0;

tic

dim = size(Data,1)-1;       %number of pixels in image

N = 2;
for j = N:-1:1
    FULL_SCALE = j;
    
    %___________Data manipulation to reduce number of pixel in image_____________
    if partition == 1 | partition == 2
        if j == N
            fprintf('\n Partition by segmentation')
            if nargin == 3
                [Data_seg, Seg_mask] = Data_segmentation(Data, partition);      %segments the data as function of intensity
            else
                load (fullfile(pathstr,seg_file));      %The segmentation data are loaded from a file
            end
            [Data_corr, T_corr, Ampl, offset] = array_corr(Data_seg); %normalize the data to insure better fitting 
            coef_fit = guess_ini_IC(Data_corr);       %guess initial coefficients
            Nb_seg = size(Data_corr,1)-1;
            F_test = F_eval(Data, IR, coef_fit, partition, Seg_mask);
            
        else
            coef_fit = desegment_coef(coef_fit, Seg_mask);      %expand coeficient vector to fit larger image 
            nouse = D1_to_D2(coef_fit, 3);
            pause(1)
            [Data_corr, T_corr, Ampl, offset] = array_corr(Data); %normalize the data to insure better fitting 
        end
    end
    
    if partition == 3
        fprintf('\n Image average')
        if j == N
            
            Data_seg(1,:) = Data(1,:);
            Data_seg(2,:) = sum(Data(2:dim+1,:),1);
            [Data_corr, T_corr, Ampl, offset] = array_corr(Data_seg); %normalize the data to insure better fitting 
            coef_fit = guess_ini_IC(Data_corr);
            Seg_mask = ones(sqrt(dim));
            F_test = F_eval(Data, IR, coef_fit, partition, Seg_mask);
            Nb_seg = 1;
            
        else
            
            [Data_corr, T_corr, Ampl, offset] = array_corr(Data); %normalize the data to insure better fitting 
            coef_ini = coef_fit;
            coef_fit = guess_ini_IC(Data_corr);
            coef_fit(1:2) = coef_ini(1:2);
        end
    end
    
    if partition == 4
        if j == N
            fprintf('\nQuadrant average')
            
            Data_seg(1,:) = Data(1,:);
            Nb_seg = 4;
            Nb_pix_sum = dim/Nb_seg;
            Seg_mask = zeros(sqrt(dim),sqrt(dim),Nb_seg);
            Seg_ones = ones(sqrt(dim)/Nb_seg, sqrt(dim));
            for i = 1:Nb_seg
                Data_seg(i+1,:) = sum(Data((i-1)*Nb_pix_sum+2:i*Nb_pix_sum+1,:),1);
                Seg_mask((i-1)*sqrt(dim)/Nb_seg+1:i*sqrt(dim)/Nb_seg,:,i) = Seg_ones;
            end
            [Data_corr, T_corr, Ampl, offset] = array_corr(Data_seg); %normalize the data to insure better fitting 
            coef_fit = guess_ini_IC(Data_corr);  
            F_test = F_eval(Data, IR, coef_fit, partition, Seg_mask);
        else
            coef_fit = desegment_coef(coef_fit, Seg_mask);      %expand coeficient vector to fit larger image 
            nouse = D1_to_D2(coef_fit, 20);
            pause(1)
            [Data_corr, T_corr, Ampl, offset] = array_corr(Data); %normalize the data to insure better fitting 
        end
    end
    
    %_________________prepares Data for fitting____________
    
    [Data_rem0, coef_fit, Pix_0] = Array_remove0(Data_corr, coef_fit, Ampl);  %Removes from data set all pixel with low intensity
    bounds = set_bound_IC(length(coef_fit));   %set_boundaries   
    Vide = [];
    exitflag = 0;
    IT = 0;
    PCG = 0;
    if j == 1   %allows to use different fitting caracteristics for the last round
        LScale = 'on';
        disp = 'off';
        num_cycle = 5;
        Tol = 1e-13;
        Fct_eval = 1e2;
    else
        LScale = 'on';
        disp = 'off';
        num_cycle = 5;
        Tol = 1e-15;
        Fct_eval = 1e3;
    end
    
    %_________calculates the error for each dat point________
    if j == N
        ERR = calc_err(Data_seg, Pix_0, Ampl);
    else
        ERR = calc_err(Data, Pix_0, Ampl);
    end
    
    %_______Start optimization_________
    fprintf('\n\n starting optimization level %g: \n', j) 
    for i = 1:num_cycle
        
        if exitflag == 0 
            %_____fmincon fitting routine_______
            options = optimset('Display', disp, 'LargeScale', LScale,...
                'GradObj', 'on', 'Hessian', 'on', 'PrecondBandWidth', 2, ...
                'MaxFunEval', Fct_eval, 'MaxIter', 1e6, 'TolFun', Tol, 'TolX', Tol);
            warning off MATLAB:divideByZero
            [coef_fit,fval,exitflag, output, lamda, grad_fit, hess_fit] = fmincon(@dblexpconv_fgh, coef_fit, Vide, Vide, Vide, Vide,...
                bounds(:,1), bounds(:,2),Vide,options, Data_rem0, IR);
            warning on    
            fprintf('\nFIT: %g', i)
            fprintf('\n T1 = %4.3e, T2 = %4.3e and LS = %4.3e\n', (1/coef_fit(1)*T_corr(2)), (1/coef_fit(2)*T_corr(2)), fval) 
            IT = IT + output.iterations;
            PCG = PCG + output.cgiterations;
            chrono(N-j+1) = toc;
            if (floor(chrono(N-j+1)/3600)) == 0
                fprintf(' %.0f min %.0f sec \n', floor(chrono(N-j+1)/60), rem(chrono(N-j+1), 60))
            else
                fprintf('%.0f h %.0f min %.0f sec \n', floor(chrono(N-j+1)/3600), floor(rem(chrono(N-j+1),3600)/60), rem(chrono(N-j+1), 60))
            end
            pause(1)
            
            if i == 1
                coef_fit = dblexp_order(coef_fit);
            end
        end
    end
    if exitflag > 0
        fprintf('\n\n Optimization terminated successfully')
        fprintf('\n Total fct call: %g Nb of iterations: %g and PCG iterations: %g \n', ITER, IT, PCG)
        
    else
        fprintf('\n\n Number of iteration not sufficient\n')
    end
    
    clear Data_simpl Data_corr bounds
    
    %___________Fit representation and output data calculated___ 
    [Data_corr, coef_fit ] = Array_replace0(Data_rem0, coef_fit, Pix_0);
    F_test = F_eval(Data, IR, coef_fit, partition, Seg_mask);
    
    
    pause(0.5)
    if j == N
        Data_fit = Fit_result_CONV(coef_fit, Data_corr(1,:), IR, T_corr, offset, Ampl);
    end
    
end
F = D1_to_D2(coef_fit, 4);
Data_fit = Fit_result_CONV(coef_fit, Data_corr(1,:), IR, T_corr, offset, Ampl);
coef_fit(1) = T_corr(2)/coef_fit(1);
coef_fit(2) = T_corr(2)/coef_fit(2);

iter_evol = ITER_EVOL(1:ITER,:);
iter_evol(:,1) = iter_evol(:,1) - iter_evol(1,1) + 1;


save coef_fit Data Data_fit chrono
%rmpath(fullfile(pathstr,'Utilities'))


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                       %
%____________________ SUB-ROUTINES _____________________%
%                                                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%______________Generate initial guess__________

function F = guess_ini_IC(Data)
% generates initial guess for fit
T_pts = size(Data,2);           %number of time points
Nb_pix = size(Data,1)-1 ;        %Number of pixels

coef0 = rand(2*Nb_pix+2,1);
coef0(1) = 2/Data(1,T_pts);
coef0(2) = 4/Data(1,T_pts);
coef0(3:2:2*Nb_pix+1) = max(Data(2:Nb_pix+1,:)');
F = coef0';

%________Set bounds for fitting__________

function F = set_bound_IC(L)
% generates upper and lower bounds for coefficients of length L

lb(1:L) = 0;
ub(1:L) = 1e1;

ub(1:2) = 1e3;
ub(4:2:L) = 1;


F(:,1) = lb';
F(:,2) = ub';

%_________________conv_2D_________________

function F = conv_2D(decay_2D, IR)
% calculates the convolution between IR and each line of the decay matrix

Nb_pix = size(decay_2D,1);
T_points = length(IR);
F = zeros(Nb_pix,T_points);

for i = 1:T_points
    for k = 1:i
        F(:,i) = F(:,i) + IR(k).*decay_2D(:,i-k+1);
    end
end

%_____________________D1_to_D2________________

function F = D1_to_D2(coef, f);
% plots output ratio coefficients in a 2D image
L = length(coef);
pix_num = (L-2)/2;

n = ceil(sqrt(pix_num));
k = 1;

for i = 1:n 
    for j = 1:n
        if (2*k+2) <= L
            S(i,j) = 1-coef(2*k+2);
        else
            S(i,j) = 1;
        end
        k = k+1;
    end
end

F = S;

figure(f+1)
imagesc(F,[0, 1])


colormap(hot)
h = gca;
set(h, 'XTick', [])
set(h, 'YTick', [])

%________________Array_corr______________________

function [F, T_corr, Ampl, offset] = array_corr(Data)
%Normalises the array between 0 and 1 for the intensity and corrects the
%time to be comprised between 1 and Nb of time points


L = size(Data,2);           %number of time points
K = size(Data,1)-1 ;        %Number of pixels
Ampl_1 = zeros(1,K);
T_corr(1) = Data(1,1);
Data(1,:) = Data(1,:) -Data(1,1);
T_corr(2) = Data(1,2) - Data(1,1);
Data(1,:) = Data(1,:)/T_corr(2);


[max_I,max_I_N]  =  max(Data(2:K+1,:),[],2);

Low_I(1:K) = mean(Data(2:K+1,1:2)');

High_I(1:K) = Data(2:K+1,round(mean(max_I_N)));

Ampl = (High_I - Low_I)';

A_0 = find(Ampl == 0);
Ampl_1 = Ampl;
Ampl_1(A_0) = 1;

offset = Low_I;
for i = 1:L
    Data(2:K+1,i) = Data(2:K+1,i) - offset'; 
    Data(2:K+1,i) = Data(2:K+1,i)./Ampl_1(1:K);
end


F = Data;


%_______________________Array_remove0_________________

function [Data_rem0, coef_rem0, Pix_0] = Array_remove0(Data, coef, Ampl)
% Removes pixels from fitting which are lower that a certain threshold
L = size(Data,1)-1;         %Number of pixels
M = size(Data,2);
j = 1;
k = 0;
Data_rem0(1,:) = Data(1,:);
coef_rem0(1:2) = coef(1:2);

if nargin == 2
    Ampl = max(Data(2:L+1, :),[],2);
end

Max_or_Tot = 0;     % 1 if max intensity based removal; 0 for total counts based removal
if Max_or_Tot == 1 
    low_lim = 0.005*(max(Ampl));
else
    Ampl_2D = Ampl*ones(1,M);
    Data_RI = Ampl_2D.*Data(2:L+1,1:M);
    Sum_data = sum(Data_RI(1:L,:),2);
    Ampl = Sum_data;
    Max_SD = max(Sum_data);
    low_lim = 500;
end


Pix_0 = find(Ampl < low_lim);
Pix_int = find(Ampl >= low_lim);
k = length(Pix_int);
coef_rem0(3:2:2*k  + 1) = coef(Pix_int*2+1);
coef_rem0(4:2:2*k + 2) = coef(Pix_int*2+2);
Data_rem0(2:k + 1,:) = Data(Pix_int+1,:);

% fprintf('\nTotal pixels: %d, number removed: %d\n', L, length(Pix_0))

%___________________Array_replace0__________________________

function [Data_rep0, coef_rep0 ] = Array_replace0(Data_rem0, coef_rem0, Pix_0)
%replaces low intensity pixels in the image that were removed befor the fit

if isempty(Pix_0)
    Data_rep0 = Data_rem0;
    coef_rep0 = coef_rem0;
else
    
    T_points = size(Data_rem0,2);
    L = size(Data_rem0,1)-1;         %Number of pixels
    K = length(Pix_0);
    L_tot = L + K;
    j = 1;
    k = 1;
    Data_rep0 = zeros(L_tot+1,T_points);
    coef_rep0 = zeros(2*L_tot+2,1);
    Pix_int = zeros(L,1);
    Data_rep0(1,:) = Data_rem0(1,:);
    coef_rep0(1:2) = coef_rem0(1:2);
    
    Pix_tot = [1:L_tot];
    Pix_tot(Pix_0) = 0;
    Pix_int = find(Pix_tot ~= 0);
    Pix_int_length = length(Pix_int);
    coef_rep0(Pix_int*2+1) = coef_rem0(3:2:L*2+1);
    coef_rep0(Pix_int*2+2) = coef_rem0(4:2:L*2+2);
    Data_rep0(Pix_int+1,:) =  Data_rem0(2:L+1,:);
    
end

%__________________________calc_err__________________________
function ERR = calc_err(Data, Pix_0, Ampl);
%calculates the error matrix 
global ERR

Nb_pix = size(Data,1)-1 ;       %Number of pixels
T_points = size(Data,2);    %Number of time points
K = length(Pix_0);          % number of pixel removed
Pix_int = zeros(Nb_pix-K,1);
k = 0;
j = 1;

ERR_tot = zeros(Nb_pix, T_points);
ERR = zeros(Nb_pix-K, T_points);
Ampl_2D = zeros(Nb_pix-K, T_points);


ERR_tot = abs(Data(2:Nb_pix+1,:));      % TCSPC has poisson noise => err = sqrt(I) qui^2 = (Imod-Idata)^2/ err^2

All_pix = [1:Nb_pix];
if isempty(Pix_0)
    Pix_int(1:Nb_pix) = All_pix;
else
    All_pix(Pix_0) = 0;
    Pix_int = find(All_pix ~= 0);
end

for i = 1:T_points
    Ampl_2D(1:Nb_pix-K, i) = Ampl(Pix_int);
end

Ampl_rem0 = Ampl(Pix_int);


ERR(1:Nb_pix-K,:) = ERR_tot(Pix_int,:);


for j = 1:T_points
    ERR(:,j) = ERR(:,j)./Ampl_2D(:,j);
    [K_0,L_0] = find(ERR(:,j) == 0);
    ERR(K_0,j) = 1./Ampl_rem0(K_0);
end

%______________________desegment_coef___________________

function coef_out = desegment_coef(coef_in, Seg_mask);
%repalce fitting coefficients of each segment in whole image coefficients
im_size = size(Seg_mask,1);
Nb_pix = im_size^2;
Nb_seg = size(Seg_mask,3);
coef_out = zeros(2*Nb_pix+2,1);
coef_out(1:2) = coef_in(1:2);
Seg_col = zeros(Nb_pix,1);
for i = 1:Nb_seg
    Seg_col(1:Nb_pix) = im2col(Seg_mask(:,:,i)',[1 1]);
    Non_Z = find(Seg_col);
    coef_out(2*Non_Z+1) = coef_in(2*i+1);       %pas sur que ca marche!!!
    coef_out(2*Non_Z+2) = coef_in(2*i+2);
end

%___________________F_eval________________________
function F = F_eval(Data, IR, coef_fit, partition, Seg_mask);
% evaluates qui_square value for given set of coefficients
global ERR ITER ITER_EVOL T_START FULL_SCALE

T_pts = size(Data,2);
Nb_pix = size(Data,1)-1;
lg_im = sqrt(Nb_pix);
Nb_coef = length(coef_fit);
red_im = sqrt((Nb_coef-2)/2);


%_____expand coefficients for large scale image___________
if partition == 0
    coef3_red = col2im(coef_fit(3:2:Nb_coef-1),[1 1],[red_im red_im]);
    coef4_red = col2im(coef_fit(4:2:Nb_coef),[1 1],[red_im red_im]);
    
    red_fact = lg_im/red_im;
    
    Nb_1 = red_im;
    Nb_2 = lg_im;
    s = ones(lg_im,1);
    j = [1:lg_im];
    
    i =floor( [1:red_im/lg_im:red_im+1]);
    i = i(1:lg_im);
    
    exp_M = sparse(i,j,s,Nb_1,Nb_2);
    coef3_lg = exp_M'*coef3_red*exp_M;
    coef4_lg = exp_M'*coef4_red*exp_M;
    
    coef_exp = zeros(2*Nb_pix+2,1);
    coef_exp(1:2) = coef_fit(1:2);
    coef_exp(3:2:2*Nb_pix+1) = im2col(coef3_lg,[1 1]);
    coef_exp(4:2:2*Nb_pix+2) = im2col(coef4_lg,[1 1]);
else
    if FULL_SCALE ~= 1
        coef_exp = desegment_coef(coef_fit, Seg_mask);
    else
        coef_exp = coef_fit;
    end
end

%______Calculates least-square with coef_exp_______

[Data_corr, T_corr, Ampl, offset] = array_corr(Data);
[Data_rem0, coef_exp, Pix_0] = Array_remove0(Data_corr, coef_exp, Ampl);
full_scale = FULL_SCALE;
err = ERR;
ERR = calc_err(Data, Pix_0, Ampl);
FULL_SCALE = 1;
[F] = dblexpconv_fgh(coef_exp, Data_rem0, IR);
ERR = err;
FULL_SCALE = full_scale;


%_______________________Fit_result_conv_________________

function F = Fit_result_CONV(coef, X, IR, T_corr, offset, Ampl)
%Generates the whole fitted array using the coefficients obtained from the fit
Nb_pix = (length(coef)-2)/2;        %Number of pixels
T_points = length(X);    %Number of time points
L_coef = length(coef);
[Tmax, Tmax_i] = max(IR);  %finds Time zero indice
x_ext = [0:T_points+Tmax_i-1]./(X(2) - X(1)) - X(1);
x_ext = x_ext - x_ext(Tmax_i);
A_2D = zeros(Nb_pix, T_points + Tmax_i);


coef_3 = coef(3:2:L_coef-1);
coef_4 = coef(4:2:L_coef);

for i = Tmax_i:T_points+Tmax_i
    A_2D(:,i) = coef_3(:).*(coef_4(:).*exp(-coef(1)*x_ext(i)) + (1-coef_4(:)).*exp(-coef(2)*x_ext(i)));
end

Y_fit = conv_2D(A_2D(:,Tmax_i:T_points+Tmax_i), IR);

for i = 1:T_points
    Y_fit(1:Nb_pix,i) = Y_fit(1:Nb_pix,i).*Ampl(1:Nb_pix);
    Y_fit(1:Nb_pix,i) = Y_fit(1:Nb_pix,i) + offset(1:Nb_pix)';   
end
F(1,1:T_points) = X*T_corr(2) + T_corr(1);
F(2:Nb_pix+1,1:T_points) = Y_fit;

function F = dblexp_order(coef)
%F: return coefficients containing coef 1 to x with ordered exponentials
%coefficients with coef 1 to x 

L = length(coef);
[coef(1:2), Index] = sort(coef(1:2));

if Index(1) == 2
    coef(4:2:L) = 1 - coef(4:2:L);
    fprintf('\nexp_order\n')
end

F = coef;