/*
  Generate a large number of pseudorandom bytes from seed bytes read
  from standard input.

  Uses the same random number as GNU libc (libc6, 2.27) (called
  "trinomials", actually lagged Fibonacci), but keeping the least
  significant bit.  (As such, the randomness is not very good.)

  A nice throughput demonstration:

  dd if=/dev/urandom count=1 | ./randdata4 -1 | dd of=/dev/null bs=10000000 status=progress

  Copyright 2020 Ken Takusagawa

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// this is based on randdata2.cpp.  I think randdata3 is broken, also
// http://www.mit.edu/~kenta/three/randdata/upuplxco/ says randdata3
// is slow.

#include <cstdio>
#include <iostream>
using std::cerr;
using std::endl;
#include <cstdlib>
#include <cstring>
#include <cassert>

//type "long" will do the right thing depending on word size
typedef long MyInt;

//An inlineable C++ implentation of the
//TYPE_3 GNU Libc random number generator
//stripped down, a lot.
struct Randclass{
  static const int DEG=31;
  static const int SEP=3;

  MyInt randtable[DEG];
  int p;

  Randclass()
    :p(0)
  {
    //initialize from stdin
    int code=fread(randtable,sizeof(MyInt),DEG,stdin);
    if(code!=DEG){
      cerr << "error: cannot initialize: "<< code << endl;
      exit(1);
    }
    bool nonzero_found=false;
    bool oddfound=false;
    for(int i=0;i<DEG;++i){
      if(randtable[i]){
        nonzero_found=true;
      }
      if(randtable[i]&1){
        oddfound=true;
      }
    }
    if(!nonzero_found){
      cerr << "error: only read zeros in initialization"<< endl;
      exit(1);
    }
    if(!oddfound){
      cerr << "error: only read even numbers in initialization"<< endl;
      exit(1);
    }
  }


  void generate_more(){
      for(int i=0;i<SEP;++i){
	int j=i+(DEG-SEP);
        assert(j<DEG);
	randtable[i]+=randtable[j];
      }
      for(int i=SEP;i<DEG;++i){
        int j=i-SEP;
	assert(j>=0);
	randtable[i]+=randtable[j];
      }
  }
};


int main(int argc, char**argv){
  const int BUFSIZE=1000000;
  if(argc!=2){
    cerr << argv[0] << " numblocks" << endl;
    cerr << endl << "    Initializes from stdin, then:" << endl;
    cerr <<         "    Emits numblocks * " << BUFSIZE << " bytes" << endl;
    cerr <<         "      If numblocks < 0, then write infinite." << endl;
    return -1;
  }
  Randclass rand;
  long long count=atoll(argv[1]);
  bool infinite = (count < 0);

  const int numtables=1+BUFSIZE/sizeof(rand.randtable); //overshoot by 1, drop data
  const int arraysize=numtables*Randclass::DEG;
  MyInt buf[arraysize];

  cerr << "sizeof(randtable)=" << sizeof(rand.randtable) << " numtables=" << numtables <<" count="<<count<<" sizeof="<<sizeof(buf)<<endl;
  for(long long ct=0;infinite || (ct<count);++ct){
    for(int i=0;i<arraysize;i+=Randclass::DEG){
      rand.generate_more();
      memcpy(&buf[i],rand.randtable,sizeof(rand.randtable));
      //additional speed might be possible by doing in place instead of memcpy
    }

    // this is where data gets dropped because BUFSIZE, not arraysize.
    // assume fwrite BUFSIZE,1 is faster than 1,BUFSIZE .
    int code=fwrite(buf,BUFSIZE,1,stdout);
    if(code!=1) {
      cerr << "write failed: " << code << endl;
      return -1; //disk full
    }
  }
}
