function [xb,yb,LB_hist] = ellip_abc_BR_rpn(p,q,h,H0,x0,r,max_iter)
% function [xb,yb_LBhist] = ellip_abc_BR_rpn(p,q,h,H0,x0,r,max_iter)
%
% ellipsoid method for solving optimal maximal "real part norm" MOR problem
% solving min F(x) = (max(abs(p*[1 ; x])./q*[1 ; x])) s.t. q*[1 ; x] > 0 for all 0 < t < pi
%
% Inputs:
%   p --- complex numerator of the objective function
%   q --- real denominator of the objective function
%   H0,x0 --- initial ellipsoid defined by E0 = {x:||(x-x0)*inv(H0)||<1} in R^n
%   r --- desirable relative accuracy of the algorithm (reduction is within
%   (1-r)*100 percent accurate, 0 < r < 1
%   max_iter --- maximum number iteration allowed
%
% Outputs:
%   xb --- the current best solution
%   yb --- the current best objective value
%   LB_hist --- history of the lower bound of objective value

% parameters
if nargin < 6
    r = 0.9;
end
if nargin < 7
    max_iter = 30000;
end

% initialization
xb = x0;    % current best x
yb = max(abs(p*[1 ; xb])./(q*[1 ; xb]));    % current best F(x)
xk = x0;    % current of the current ellipsoid Ek
Hk = H0;    % matrix of the current ellipsoid Ek
LB_hist = zeros(max_iter,1);

% main iteration
for iter = 1:max_iter
    y = r*yb;   % current superoptimal level
    % add superoptimal cut
    [L,ds] = obj_cut(p,q,xk,y);
    if ds ~= -1 % add superoptimal cut and update ellipsoid (or verify that the ellipsoid is empty)
        % check whether ellipsoid is empty or not
        nrm = norm(L*Hk);
        if nrm <= ds
            disp('empty objective cut');
            LB_hist = LB_hist(1:iter);
            return
        end
        [Hk,xk]=ellips_cut1(Hk,xk,L,ds/nrm);       
    else
        % check feasiblity of xk
        [L,ds] = br_check_descarte(xk,1/h);
        %[L,ds] = br_check(xk);
        if ds == -inf   % xk is feasible and upadate xb (yb)
            xb = xk;
            yb = max(abs(p*[1 ; xk])./(q*[1 ; xk]));
        else    % xk is infeasible, check feasibility of xt = 0.5*(xb+xk)
            xt = 0.5*(xb+xk);
            %[L1,ds1] = br_check_hamiltonian(xt,1/h);
            [L1,ds1] = br_check_descarte(xt,1/h);
            %[L1,ds1] = br_check(xt);
            if ds1 ~= -inf  % xt is infeasible, hence add feasibility cut
                L = L1;
                ds = L*(xt-xk)+ds1;
                nrm = norm(L*Hk);
                if nrm <= ds
                    disp('empty feasibility cut');
                    LB_hist = LB_hist(1:iter);
                    return
                else
                    [Hk,xk] = ellips_cut1(Hk,xk,L,ds/nrm);
                end
            else    % xt is feasible, do a line search
                [ystar,xstar] = lfo_linear_opt(p,q,xt,xb,8);
                if ystar < yb
                    xb = xstar;
                    yb = ystar;                    
                else    % add feasibility cut of xk
                    % check whether ellipsoid is empty or not
                    nrm = norm(L*Hk);
                    if nrm <= ds
                        disp('empty feasibility cut');
                        LB_hist = LB_hist(1:iter);
                        return                        
                    else
                        % add feasibility cut
                        nrm = norm(L*Hk);
                        [Hk,xk] = ellips_cut1(Hk,xk,L,ds/nrm);
                    end
                end
            end
        end
    end
    LB_hist(iter) = yb;
end

disp('exit due to iteration limit...');