function p = compute_p1w_BR(t,G,q,W)
% function p = compute_p1w_BR(t,G,q,W)
%
% compute numerator of reduced system (ellipsoid algorithm, bounded
% realness condition)
%
% t --- discrete-time frequencies
% G --- frequency response of the original system
% q --- denominator of the reduced system in decreasing order
% weigthed version of compute_p1

% preprocessing
t = t(:);
G = G(:);
nt = size(t,1);
q = q(:)';
m = size(q,2)-1;    % order of reduced system
if nargin < 4
    W = ones(nt,1);
else
    W = W(:);
end

% normalize the frequency response
h = max(abs(G));
Gn = G/h;

% compute the initial ellipsoid
z = exp(j*t);
Z = repmat(z,1,m+1).^repmat((m:-1:0),nt,1);
%Mr = real(Z);
%Mi = imag(Z);
qz = polyval(q,z);
%h0 = Gn.*qz;
%Q = Mr'*Mr+Mi'*Mi;
%b = -2*real((Mr'-j*Mi')*h0);
%tt = norm(qz,2)^2-norm(h0,2)^2;
%x0 = -(Q\b)/2;  % center of the initial ellipsoid
%H = Q/(tt-b'*inv(Q)*b/4);
%[U,S,V] = svd(H);
%H0 = diag(diag(S).^(-0.5))*U';    % matrix defining the initial ellipsoid

M = 1e5;
H0 = diag(repmat(M,m+1,1));
x0 = zeros(m+1,1);

% build some necessary matrices
P = (repmat(z,1,m+1).^repmat((m:-1:0),nt,1));

r = 0.99;
max_iter = 80000;

% call ellip_compute_p1w_PR
[p,y] = ellip_compute_p1w_BR(q,qz,W.*Gn,diag(W)*P,h,H0,x0,r,max_iter);
p = p'*h;

function [xb,yb] = ellip_compute_p1w_BR(q,qz,G,P,h,H0,x0,r,max_iter)
% function [xb,yb] = ellip_compute_p1w_BR(q,qz,G,P,h,H0,x0,r,max_iter)
%
% ellipsoid algorithm for solving positive real numerator of reduced system
% problem
% min_p |p/q-G| : p(z)q(1/z)+p(1/z)q(z) > 0 for all |z| = 1
%
% Inputs:
% q --- denominator of the reduced system
% qz2 --- abs(q(z))^2 at evaluation points
% Gqz2 --- G(z)*abs(q(z))^2 at evaluation points
% P --- p(z)*q(1/z) = P*x at evaluation points
% H0 --- initial ellipsoid = {x : |inv(H0)*(x-x0)| <= 1
% x0 --- center of the initial ellipsoid
% r --- superoptimal level
% max_iter --- maximum allowable iterations
%
% Outputs:
% x --- center of the final ellipsoid
% y --- cost associated with x

% parameters
if nargin < 7
    r = 0.9;
end
if nargin < 8
    max_iter = 30000;
end

% initialization
xb = x0;    % current best x
yb = max(abs((P*xb-G.*qz)./qz));    % current best F(x)
%xb = [];
%yb = inf;
xk = x0;    % current of the current ellipsoid Ek
Hk = H0;    % matrix of the current ellipsoid Ek
LB_hist = zeros(max_iter,1);
cut_hist = zeros(max_iter,1);

% main iteration
for iter = 1:max_iter
    y = r*yb;   % current superoptimal level
    % add superoptimal cut
    tic;
    [L,ds] = obj_cut_compute_p1(qz,G,P,xk,y);
    if ds ~= -1 % add superoptimal cut and update ellipsoid (or verify that the ellipsoid is empty)
        % check whether ellipsoid is empty or not
        nrm = norm(L*Hk);
        if nrm <= ds
            return
        end
        [Hk,xk]=ellips_cut1(Hk,xk,L,ds/nrm);
        cut_hist(iter) = 1;
    else
        % check feasiblity of xk (positive-realness)
        [L,ds] = BR_check(xk,q,1/h);
        if ds == -inf   % xk is feasible and upadate xb (yb)
            xb = xk;
            yb = max(abs((P*xk-G.*qz)./qz));
        elseif isempty(xb)  % yb is not found yet, simply add feasibility cut
            nrm = norm(L*Hk);
            if nrm <= ds
                return;
            else
                [Hk,xk] = ellips_cut1(Hk,xk,L,ds/nrm);
                cut_hist(iter) = -1;
            end
        else    % xk is infeasible, check feasibility of xt = 0.5*(xb+xk)
            xt = 0.5*(xb+xk);
            [L1,ds1] = BR_check(xt,q,1/h);
            if ds1 ~= -inf  % xt is infeasible, hence add feasibility cut
                L = L1;
                ds = L*(xt-xk)+ds1;
                nrm = norm(L*Hk);
                if nrm <= ds
                    return
                else
                    [Hk,xk] = ellips_cut1(Hk,xk,L,ds/nrm);
                    cut_hist(iter) = -1;
                end
            else    % xt is feasible, add feasibility cut of xk
                if max(abs((P*xt-G.*qz)./qz)) < yb
                    xb = xt;
                    yb = max(abs((P*xt-G.*qz)./qz));
                else
                    % check whether ellipsoid is empty or not
                    nrm = norm(L*Hk);
                    if nrm <= ds
                        return
                    else
                        % add feasibility cut
                        nrm = norm(L*Hk);
                        [Hk,xk] = ellips_cut1(Hk,xk,L,ds/nrm);
                        cut_hist(iter) = -1;
                    end
                end
            end
        end
    end
    LB_hist(iter) = yb;
end

% return output due to iteration limit
disp('exit due to iteration limit!')

function [L,ds] = obj_cut_compute_p1(qz,G,P,x,y)
% function [L,ds] = obj_cut_compute_p1(qz,G,P,x,y)
%
% objective cut function

m = length(x)-1;
nmd = abs(P*x-G.*qz)-y*abs(qz);
[nmd_max,max_ind] = max(nmd);
if nmd_max < 0
    L = zeros(1,m+1);
    ds = -1;
else
    w = exp(-j*angle((P(max_ind,:)*x-G(max_ind)*qz(max_ind))/qz(max_ind)));
    L = -real(w*P(max_ind,:)/qz(max_ind));
    ds = -y-real(w*G(max_ind))-L*x;
    if ds < 0
        ds = 1e-14;
    end
end

function [L,ds] = BR_check(x,q,h)
% function [L,ds] = BR_check(x,q,h)
%
% checking bounded realness condition

if nargin < 3
    h = 1;
end

tol = 0.999;
h = tol*h;

q = q(:);
x = x(:);
m = length(x)-1;
Tz = h^2*conv(q,q(end:-1:1))-conv(x,x(end:-1:1));
Tt = [Tz(m+1) ; 2*Tz(m+2:end)];
t0 = tri_poly_pos(Tt);
if isempty(t0)
    L = zeros(1,m+1);
    ds = -inf;
else
    w = exp(-j*angle((polyval(x,exp(j*t0))/polyval(q,exp(j*t0)))));
    L = -real(w/polyval(q,exp(j*t0))*(exp(j*t0).^(m:-1:0)));
    ds = -h-L*x;
    if ds < 0
        ds = 1e-14;
    end
end