function [L,ds] = br_check_descarte(x,h)
% function [L,ds] = br_check_descarte(x,h)
%
% check stability and bounded realness (|p|/q < h) condition using
% Descarte's rule of signs
%
% if x is feasible, ds = -inf; if x is not, then return a cut such that all
% xx in R^n should be in the set {xx| L(xx-x) > ds}, and ds >= 0

if nargin < 2
    h = 1;
end

tol = 1e-6;
tol_br = 0.999;
h = tol_br*h;
x = x(:);
n = length(x);
m = floor(n/3);
xa = x(1:m);
xb = x(m+1:2*m+1);
xc = x(2*m+2:3*m+1);
a = [1 ; 0.5*xa];
b = [xb(1) ; 0.5*xb(2:end)];
c = 0.5*xc;
az = [a(end:-1:2) ; a(1) ; a(2:end)];
bz = [b(end:-1:2) ; b(1) ; b(2:end)];
cz = [c(end:-1:1) ; 0 ; -c];

% check stability condition first
root_a = roots(az);
[p_margin,ind] = min(abs(abs(root_a)-1));
if p_margin <= tol  % stability condition is violated
    t0 = angle(root_a(ind));
    L = [cos((1:m)*t0) zeros(1,2*m+1)];
    ds = -1-L*x;
else    % stability condition is satisfied, check bounded realness condition
    P = (h^2)*conv(az,az)-conv(bz,bz)+conv(cz,cz);
    Pt = [P(2*m+1) ; 2*P(2*m+2:end)];
    t0 = tri_poly_pos(Pt);
    if ~isempty(t0) % bounded realness condition is violated at t0
        w = exp(-j*angle((cos((0:m)*t0)*xb)+j*sin((1:m)*t0)*xc));
        L = [h*cos((1:m)*t0) -real(w*cos((0:m)*t0)) -real(j*w*sin((1:m)*t0))];
        ds = -h-L*x;
    else
        L = zeros(1,n);
        ds = -inf;
    end
end