function [L,ds] = br_check(x,h)
% function [L,ds] = br_check(x,h)
%
% check stability and bounded realness (|p|/q < h) condition
%
% if x is feasible, ds = -inf; if x is not, then return a cut such that all
% xx in R^n should be in the set {xx| L(xx-x) > ds}, and ds >= 0

if nargin < 2
    h = 1;
end

tol = 1e-6;
x = x(:);
n = length(x);
m = floor(n/3);
xa = x(1:m);
xb = x(m+1:2*m+1);
xc = x(2*m+2:3*m+1);
a = [1 ; 0.5*xa];
b = [xb(1) ; 0.5*xb(2:end)];
c = 0.5*xc;
az = [a(end:-1:2) ; a(1) ; a(2:end)];
bz = [b(end:-1:2) ; b(1) ; b(2:end)];
cz = [c(end:-1:1) ; 0 ; -c];

% check stability condition first
root_a = roots(az);
[p_margin,ind] = min(abs(abs(root_a)-1));
if p_margin <= tol  % stability condition is violated
    t0 = angle(root_a(ind));
    L = [cos((1:m)*t0) zeros(1,2*m+1)];
    ds = -1-L*x;
    return;
else    % stability condition is satisfied, check bounded realness condition
    P = (h^2)*conv(az,az)-conv(bz,bz)+conv(cz,cz);       
    root_P = roots(P);
    [f_margin,ind] = min(abs(abs(root_P)-1));
    if f_margin > tol   % no zero axis crossing
        if P(2*m+1) <= 0    % bounded realness condition is violated at EVERY point
            t0 = 0.5*pi;    % quite arbitrary
            w = exp(-j*angle((cos((0:m)*t0)*xb)+j*sin((1:m)*t0)*xc));
            L = [h*cos((1:m)*t0) -real(w*cos((0:m)*t0)) -real(j*w*sin((1:m)*t0))];
            ds = -h-L*x;
            return;
        else    % bounded realness condition is satisfied
            L = zeros(1,n);
            ds = -inf;
            return;
        end
    else    % bounded realness condition is violated (in particular, at zero crossing frequency)
        t0 = angle(root_P(ind));
        w = exp(-j*angle((cos((0:m)*t0)*xb)+j*sin((1:m)*t0)*xc));
        L = [h*cos((1:m)*t0) -real(w*cos((0:m)*t0)) -real(j*w*sin((1:m)*t0))];
        ds = -h-L*x;
    end
end